/*
 * Copyright (C) 2000 - 2024 Silverpeas
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * As a special exception to the terms and conditions of version 3.0 of
 * the GPL, you may redistribute this Program in connection with Free/Libre
 * Open Source Software ("FLOSS") applications as described in Silverpeas's
 * FLOSS exception.  You should have received a copy of the text describing
 * the FLOSS exception, and it is also available here:
 * "https://www.silverpeas.org/legal/floss_exception.html"
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.silverpeas.core.viewer.model;

import org.silverpeas.kernel.SilverpeasException;
import org.silverpeas.core.io.media.image.ImageTool;
import org.silverpeas.core.util.ImageUtil;
import org.silverpeas.core.util.file.FileServerUtils;
import org.silverpeas.kernel.logging.SilverLogger;
import org.silverpeas.core.viewer.service.ViewerContext;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;

import static org.silverpeas.core.io.media.image.ImageInfoType.HEIGHT_IN_PIXEL;
import static org.silverpeas.core.io.media.image.ImageInfoType.WIDTH_IN_PIXEL;
import static org.silverpeas.core.util.file.FileRepositoryManager.getTemporaryPath;

/**
 * @author Yohann Chastagnier
 */
public abstract class AbstractPreview implements Preview {
  private static final long serialVersionUID = 3597757215012779572L;

  private final String documentId;
  private final String documentType;
  private final String language;
  private final String originalFileName;
  private final File physicalFile;
  private String[] widthAndHeight = null;

  /*
   * (non-Javadoc)
   * @see org.silverpeas.core.viewer.model.Preview#getDisplayLicenseKey()
   */
  @Override
  public String getDisplayLicenseKey() {
    return "";
  }

  /**
   * Default constructor
   */
  AbstractPreview(final ViewerContext context, final File physicalFile) {
    this.documentId = context.getDocumentId();
    this.documentType = context.getDocumentType();
    this.language = context.getLanguage();
    this.originalFileName = context.getOriginalFileName();
    this.physicalFile = physicalFile;
  }

  @Override
  public String getDocumentId() {
    return this.documentId;
  }

  @Override
  public String getDocumentType() {
    return this.documentType;
  }

  @Override
  public String getLanguage() {
    return this.language;
  }

  /*
   * (non-Javadoc)
   * @see org.silverpeas.core.viewer.model.Preview#getOriginalFileName()
   */
  @Override
  public String getOriginalFileName() {
    return originalFileName;
  }

  /*
   * (non-Javadoc)
   * @see org.silverpeas.core.viewer.model.Preview#getPhysicalFile()
   */
  @Override
  public File getPhysicalFile() {
    return physicalFile;
  }

  /*
   * (non-Javadoc)
   * @see org.silverpeas.core.viewer.model.Preview#getWidth()
   */
  @Override
  public String getWidth() {
    return getWidthAndHeight()[0];
  }

  /*
   * (non-Javadoc)
   * @see org.silverpeas.core.viewer.model.Preview#getHeight()
   */
  @Override
  public String getHeight() {
    return getWidthAndHeight()[1];
  }

  /*
   * (non-Javadoc)
   * @see org.silverpeas.core.viewer.model.Preview#getURLAsString()
   */
  @Override
  public String getURLAsString() {
    return FileServerUtils.getUrlToTempDir(getPhysicalFile().getParentFile().getName() + "/" +
        getPhysicalFile().getName());
  }

  @Override
  public Path getServerFilePath() {
    return Paths.get(getTemporaryPath(), getPhysicalFile().getParentFile().getName(),
        getPhysicalFile().getName());
  }

  /**
   * Centralized method providing width and height of a preview
   * @return
   */
  private String[] getWidthAndHeight() {
    if (widthAndHeight == null) {
      final ImageTool imageTool = ImageTool.get();
      if (imageTool.isActivated()) {
        try {
          widthAndHeight = imageTool.getImageInfo(physicalFile, WIDTH_IN_PIXEL, HEIGHT_IN_PIXEL);
        } catch (SilverpeasException e) {
          SilverLogger.getLogger(this).warn(e);
        }
      }
      if (widthAndHeight == null) {
        widthAndHeight = ImageUtil.getWidthAndHeight(physicalFile);
      }
    }
    return widthAndHeight;
  }
}
